<?php

namespace App\Livewire\Forms;

use Livewire\Component;
use App\Models\CustomMenu;
use Livewire\Attributes\On;
use Jantinnerezo\LivewireAlert\LivewireAlert;

class DisableLanding extends Component
{

    use LivewireAlert;

    public $settings;
    public $disableLandingSite;
    public $landingSiteType;
    public $landingSiteUrl;
    public $facebook;
    public $instagram;
    public $twitter;
    public $yelp;
    public $metaKeyword;
    public $metaDescription;
    public $metaTitle;
    public $trixId;
    public $menuName;
    public $menuSlug;
    public $menuContent;
    public $addDyanamicMenuModal = false;
    public $showEditDynamicMenuModal = false;
    public $confirmDeleteMenuModal = false;
    public $editMenuId;
    public $editMenuName;
    public $editMenuSlug;
    public $editMenuContent;
    public $showAddDynamicMenu;
    public $menuId;
    public $menu;
    public $menuStates = [];
    public $menuIdToDelete = null;

    protected $listeners = ['refreshCustomers' => '$refresh'];

    public function mount()
    {
        $this->disableLandingSite = $this->settings ? (bool)$this->settings->disable_landing_site : false;
        $this->landingSiteType = $this->settings ? $this->settings->landing_site_type : '';
        $this->landingSiteUrl = $this->settings ? $this->settings->landing_site_url : '';
        $this->facebook = $this->settings ? $this->settings->facebook_link : '';
        $this->instagram = $this->settings ? $this->settings->instagram_link : '';
        $this->twitter = $this->settings ? $this->settings->twitter_link : '';
        $this->yelp = $this->settings ? $this->settings->yelp_link : '';
        $this->metaTitle = $this->settings ? $this->settings->meta_title : '';
        $this->metaKeyword = $this->settings ? $this->settings->meta_keyword : '';
        $this->metaDescription = $this->settings ? $this->settings->meta_description : '';
        $this->trixId = 'trix-' . uniqid();
    }

    public function submitForm()
    {
        $this->validate([
            'landingSiteUrl' => [
                'required_if:landingSiteType,custom',
                'nullable',
                'url',
                function ($attribute, $value, $fail) {
                    if ($value) {
                        $host = parse_url($value, PHP_URL_HOST);
                        $appUrl = parse_url(config('app.url'), PHP_URL_HOST);

                        if ($host === $appUrl) {
                            $fail(__('messages.cannotUseSelfDomain'));
                        }
                    }
                }
            ],
        ]);

        $this->settings->disable_landing_site = $this->disableLandingSite;
        $this->settings->landing_site_type = $this->landingSiteType;
        $this->settings->landing_site_url = $this->landingSiteUrl;
        $this->settings->facebook_link = $this->facebook;
        $this->settings->instagram_link = $this->instagram;
        $this->settings->twitter_link = $this->twitter;
        $this->settings->yelp_link = $this->yelp;
        $this->settings->meta_title = $this->metaTitle;
        $this->settings->meta_keyword = $this->metaKeyword;
        $this->settings->meta_description = $this->metaDescription;


        $this->settings->save();

        cache()->forget('global_setting');

        $this->alert('success', __('messages.settingsUpdated'), [
            'toast' => true,
            'position' => 'top-end',
            'showCancelButton' => false,
            'cancelButtonText' => __('app.close')
        ]);
    }

    public function submitDynamicWebPageForm()
    {
        $this->validate([
            'menuName' => 'required',
            'menuSlug' => 'required',
            'menuContent' => 'required',
        ]);

        CustomMenu::create([
            'menu_name' => $this->menuName,
            'menu_slug' => $this->menuSlug,
            'menu_content' => $this->menuContent,
        ]);
        $this->alert('success', __('messages.settingsUpdated'), [
            'toast' => true,
            'position' => 'top-end',
            'showCancelButton' => false,
            'cancelButtonText' => __('app.close')
        ]);
    }

    public function showEditDynamicMenu($id)
    {
        $this->menuId = $id;
        $this->showEditDynamicMenuModal = true;
    }

    public function showAddDynamicMenu()
    {
        $this->addDyanamicMenuModal = true;
    }

    public function editDynamicMenu()
    {
        $this->validate([
            'editMenuName' => 'required',
            'editMenuSlug' => 'required',
            'editMenuContent' => 'required',
        ]);

        $menu = CustomMenu::findOrFail($this->menuId);
        $menu->menu_name = $this->editMenuName;
        $menu->menu_slug = $this->editMenuSlug;
        $menu->menu_content = $this->editMenuContent;
        $menu->save();

        $this->showEditDynamicMenuModal = false;

        $this->alert('success', __('messages.settingsUpdated'), [
            'toast' => true,
            'position' => 'top-end',
            'showCancelButton' => false,
            'cancelButtonText' => __('app.close')
        ]);
    }

    public function confirmDeleteMenu($menuId)
    {
        $this->menuIdToDelete = $menuId;
    }

    public function deleteMenu()
    {
        if ($this->menuIdToDelete) {
            CustomMenu::find($this->menuIdToDelete)->delete();
            $this->menuIdToDelete = null;
            $this->alert('success', __('messages.settingsUpdated'), [
                'toast' => true,
                'position' => 'top-end',
                'showCancelButton' => false,
                'cancelButtonText' => __('app.close')
            ]);
        }

        $this->dispatch('$refresh');
    }

    public function toggleMenuStatus($menuId)
    {
        $menu = CustomMenu::findOrFail($menuId);
        $menu->is_active = !$menu->is_active;
        $menu->save();

        $this->dispatch('$refresh');

        $this->alert('success', __('messages.settingsUpdated'), [
            'toast' => true,
            'position' => 'top-end',
            'showCancelButton' => false,
            'cancelButtonText' => __('app.close')
        ]);
    }

    public function render()
    {
        return view('livewire.forms.disable-landing', [
            'customMenu' => CustomMenu::paginate(10),
        ]);
    }
}
